"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const req_1 = require("kdweb-core/lib/service/req");
const utils_1 = require("kdweb-core/lib/utils");
const WechatDefine_1 = require("../../../kds-base-define/src/WechatDefine");
const config_1 = require("../config");
const log_1 = require("../log");
const service_1 = require("./service");
const md5 = require("md5");
const CryptoJS = require("crypto-js");
const rpc_1 = require("../rpc");
const async_1 = require("kdweb-core/lib/tools/async");
let kvFriendTablename = "t_wxpay_friend";
let urls = {
    cancelPay: {
        live: "https://api.weixin.qq.com/cgi-bin/midas/cancelpay?access_token=",
        company: "https://api.weixin.qq.com/cgi-bin/midas/sandbox/cancelpay?access_token=",
        uriLive: "/cgi-bin/midas/cancelpay",
        uriCompany: "/cgi-bin/midas/sandbox/cancelpay",
    },
    getBalance: {
        live: "https://api.weixin.qq.com/cgi-bin/midas/getbalance?access_token=",
        company: "https://api.weixin.qq.com/cgi-bin/midas/sandbox/getbalance?access_token=",
        uriLive: "/cgi-bin/midas/getbalance",
        uriCompany: "/cgi-bin/midas/sandbox/getbalance",
    },
    pay: {
        live: "https://api.weixin.qq.com/cgi-bin/midas/pay?access_token=",
        company: "https://api.weixin.qq.com/cgi-bin/midas/sandbox/pay?access_token=",
        uriLive: "/cgi-bin/midas/pay",
        uriCompany: "/cgi-bin/midas/sandbox/pay",
    },
    present: {
        live: "https://api.weixin.qq.com/cgi-bin/midas/present?access_token=",
        company: "https://api.weixin.qq.com/cgi-bin/midas/sandbox/present?access_token=",
        uriLive: "/cgi-bin/midas/present",
        uriCompany: "/cgi-bin/midas/sandbox/present",
    },
};
function getSign(obj, uri, secret) {
    obj = utils_1.kdutils.clone(obj);
    let keys = Object.keys(obj);
    keys.sort();
    let str = "";
    for (let k of keys) {
        if (str.length > 0) {
            str += "&";
        }
        str += k + "=" + obj[k];
    }
    str += "&org_loc=" + uri + "&method=POST&secret=" + secret;
    log_1.Log.oth.info("sign str = " + str);
    let ret = CryptoJS.HmacSHA256(str, secret).toString();
    log_1.Log.oth.info("sign = " + ret);
    return ret; // CryptoJS.enc.Base64
}
function getSimpleSign(obj, sessionKey) {
    let keys = Object.keys(obj);
    let values = [];
    for (let k of keys) {
        values.push(obj[k]);
    }
    values.sort();
    let str = values.join("");
    return CryptoJS.HmacSHA256(str, sessionKey).toString();
}
var WechatPay;
(function (WechatPay) {
    function createBillNo(gameName, userID) {
        return __awaiter(this, void 0, void 0, function* () {
            let today = utils_1.kdutils.getFmtMoment("YYYYMMDD");
            //let fullDate = kdutils.getFmtMoment("YYYYMMDDHHmmss")
            let id = yield rpc_1.Rpc.center.callException("kds.ids.getId", "wechat-pay-billno", 0, 1);
            return today + "|" + id;
        });
    }
    WechatPay.createBillNo = createBillNo;
    function prepareSelfPay(gameName, userID, money, sandBox) {
        let config = config_1.Config.localConfig.games.find(v => v.gameName == gameName);
        if (config == null || config.pay == null) {
            return null;
        }
        let ret = {
            mode: "game",
            env: sandBox ? 1 : 0,
            offerId: config.pay.appID,
            currencyType: "CNY",
            platform: "android",
            buyQuantity: money,
            zoneId: config.pay.zoneID
        };
        return ret;
    }
    WechatPay.prepareSelfPay = prepareSelfPay;
    function prepareOtherPay(gameName, userID, money, billID, callbackRpcMethod, sandBox) {
        return __awaiter(this, void 0, void 0, function* () {
            let config = config_1.Config.localConfig.games.find(v => v.gameName == gameName);
            if (config == null || config.pay == null) {
                return null;
            }
            let sessionKey = yield service_1.WechatService.getUserSessionKeyByUserID(userID);
            if (sessionKey == null) {
                return null;
            }
            let time = utils_1.kdutils.getMillionSecond();
            let ret = {
                mode: "game",
                env: sandBox ? 1 : 0,
                offerId: config.pay.appID,
                currencyType: "CNY",
                platform: "android",
                buyQuantity: money,
                zoneId: config.pay.zoneID,
                outTradeNo: billID,
                nonceStr: md5(time),
                timeStamp: time,
            };
            let sign = getSimpleSign(ret, sessionKey);
            ret.signature = sign;
            let record = {
                gameName: gameName,
                userID: userID,
                sessionKey: sessionKey,
                timestamp: time,
                date: utils_1.kdutils.getFmtMoment("YYYY-MM-DD", time),
                fullDate: utils_1.kdutils.getFmtMoment("YYYY-MM-DD HH:mm:ss", time),
                callbackRpcMethod: callbackRpcMethod,
                sandBox: sandBox ? true : false,
                prepare: ret
            };
            rpc_1.Rpc.center.callException("kds.dbp.kv.m.insert", kvFriendTablename, record);
            return ret;
        });
    }
    WechatPay.prepareOtherPay = prepareOtherPay;
    function getAccount(gameName, userID, sandBox) {
        return __awaiter(this, void 0, void 0, function* () {
            let openID = yield service_1.WechatService.getUserOpenID(userID);
            if (openID == null) {
                log_1.Log.oth.error("[pay]getAccount: cannot find openID userID = " + userID);
                return null;
            }
            let config = config_1.Config.localConfig.games.find(v => v.gameName == gameName);
            if (config == null || config.pay == null) {
                return null;
            }
            let ak = yield service_1.WechatService.getAK(gameName);
            if (ak == null) {
                log_1.Log.oth.error("[pay]getAccount: get ak failed gameName = " + gameName);
                return null;
            }
            let url = sandBox ? urls.getBalance.company : urls.getBalance.live;
            let uri = sandBox ? urls.getBalance.uriCompany : urls.getBalance.uriLive;
            let secret = sandBox ? config.pay.appKeyCompany : config.pay.appKeyLive;
            let data = {
                openid: openID,
                appid: config.appId,
                offer_id: config.pay.appID,
                ts: Math.floor(utils_1.kdutils.getMillionSecond() / 1000),
                zone_id: config.pay.zoneID,
                pf: "android",
            };
            let sign = getSign(data, uri, secret);
            data["sig"] = sign;
            log_1.Log.oth.info("[pay]getAccount: req url = " + url + " data = ", data);
            let res = yield req_1.kdreq.postJson(url + ak, data);
            if (res.error || res.body == null) {
                log_1.Log.oth.error("[pay]getAccount: call wechat failed error = ", res.error);
                return null;
            }
            let body = res.body;
            if (body.errcode != 0) {
                log_1.Log.oth.error("[pay]getAccount: call wechat failed ret error = ", body);
                return null;
            }
            log_1.Log.oth.info("[pay]getAccount body = ", body);
            let account = {
                userID: userID,
                gameName: gameName,
                ratio: config.pay.ratio,
                money: body.balance / config.pay.ratio,
                coinCount: body.balance,
                giveCoinCount: body.gen_balance,
                firstCharge: body.first_save,
                totalChargeCoin: body.save_amt,
                totalMoney: body.save_sum,
                totalCostMoney: body.cost_sum,
                totalGiveMoney: body.present_sum,
            };
            return account;
        });
    }
    WechatPay.getAccount = getAccount;
    function payMoney(gameName, userID, money, content, sandBox) {
        return __awaiter(this, void 0, void 0, function* () {
            let openID = yield service_1.WechatService.getUserOpenID(userID);
            if (openID == null) {
                log_1.Log.oth.error("[pay]payMoney: cannot find openID userID = " + userID);
                return WechatDefine_1.WechatDefine.Pay.PayCode.WrongUserID;
            }
            let config = config_1.Config.localConfig.games.find(v => v.gameName == gameName);
            if (config == null || config.pay == null) {
                return WechatDefine_1.WechatDefine.Pay.PayCode.WrongArgs;
            }
            let ak = yield service_1.WechatService.getAK(gameName);
            if (ak == null) {
                log_1.Log.oth.error("[pay]payMoney: get ak failed gameName = " + gameName);
                return WechatDefine_1.WechatDefine.Pay.PayCode.AKFailed;
            }
            let account = yield getAccount(gameName, userID, sandBox);
            let count = money * config.pay.ratio;
            if (count > account.coinCount) {
                log_1.Log.oth.error("[pay]payMoney not enough need = " + count + " account = ", account);
                return WechatDefine_1.WechatDefine.Pay.PayCode.NotEnough;
            }
            let billNo = yield createBillNo(gameName, userID);
            if (billNo == null) {
                return WechatDefine_1.WechatDefine.Pay.PayCode.BillNoFailed;
            }
            let url = sandBox ? urls.pay.company : urls.pay.live;
            let uri = sandBox ? urls.pay.uriCompany : urls.pay.uriLive;
            let secret = sandBox ? config.pay.appKeyCompany : config.pay.appKeyLive;
            let data = {
                openid: openID,
                appid: config.appId,
                offer_id: config.pay.appID,
                ts: Math.floor(utils_1.kdutils.getMillionSecond() / 1000),
                zone_id: config.pay.zoneID,
                pf: "android",
                amt: count,
                bill_no: billNo,
                app_remark: content,
            };
            let sign = getSign(data, uri, secret);
            data["sig"] = sign;
            let tryCount = 0;
            while (true) {
                let res = yield req_1.kdreq.postJson(url + ak, data);
                if (res.error || res.body == null) {
                    log_1.Log.oth.error("[pay]payMoney: call wechat failed error = ", res.error);
                    return WechatDefine_1.WechatDefine.Pay.PayCode.ReqFailed;
                }
                let body = res.body;
                if (body.errcode != 0) {
                    if (body.errcode == -1) {
                        log_1.Log.oth.info("[pay]payMoney: system busy, try count = " + tryCount);
                        tryCount++;
                        if (tryCount >= 5) {
                            return WechatDefine_1.WechatDefine.Pay.PayCode.SystemBusy;
                        }
                        yield async_1.kdasync.timeout(1000);
                        continue;
                    }
                    log_1.Log.oth.error("[pay]payMoney: call wechat failed ret error = ", body);
                    return WechatDefine_1.WechatDefine.Pay.PayCode.ReqFailed;
                }
                break;
            }
            return WechatDefine_1.WechatDefine.Pay.PayCode.success;
        });
    }
    WechatPay.payMoney = payMoney;
})(WechatPay = exports.WechatPay || (exports.WechatPay = {}));
