"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const config_1 = require("../config");
const log_1 = require("../log");
const utils_1 = require("kdweb-core/lib/utils");
const async_1 = require("kdweb-core/lib/tools/async");
const req_1 = require("kdweb-core/lib/service/req");
const rpc_1 = require("../rpc");
const KVDefine_1 = require("../../../kds-base-define/src/KVDefine");
const _request = require("request");
let akInfos = [];
let traceList = [];
var WechatService;
(function (WechatService) {
    function getConfig(gameName) {
        return config_1.Config.localConfig.games.find(v => v.gameName == gameName);
    }
    function reqGet(url) {
        return __awaiter(this, void 0, void 0, function* () {
            let ret = yield req_1.kdreq.get(url);
            if (!ret.success || !ret.body) {
                log_1.Log.oth.error("[wechat] req failed succes == false url = " + url, ret);
                return null;
            }
            let t = null;
            try {
                t = JSON.parse(ret.body);
            }
            catch (error) {
                log_1.Log.oth.error("[wechat] req failed url = " + url, ret);
                return null;
            }
            if (t.errcode != null && t.errcode != 0) {
                log_1.Log.oth.error("[wechat] req wechat failed url = " + url + " | t = ", t);
                return null;
            }
            return t;
        });
    }
    WechatService.reqGet = reqGet;
    function getAK(gameName) {
        return __awaiter(this, void 0, void 0, function* () {
            let info = akInfos.find(v => v.gameName == gameName);
            if (info == null) {
                let config = getConfig(gameName);
                if (config == null) {
                    log_1.Log.oth.info("[wechat] cannot get config gameName = " + gameName);
                    return null;
                }
                info = {
                    gameName: gameName,
                    ak: null,
                    expire: null,
                    startTime: null,
                    waiting: true
                };
                akInfos.push(info);
            }
            else {
                if (info.waiting) {
                    let time = utils_1.kdutils.getMillionSecond();
                    while (true) {
                        yield async_1.kdasync.timeout(500);
                        if (!info.waiting) {
                            return info.ak;
                        }
                        if (utils_1.kdutils.getMillionSecond() - time > 10000) {
                            log_1.Log.oth.info("[wechat] wait ak timeout gameName = " + gameName);
                            return null;
                        }
                    }
                }
                if (info.ak != null && (utils_1.kdutils.getMillionSecond() - info.startTime) < info.expire) {
                    return info.ak;
                }
            }
            info.ak = null;
            info.waiting = true;
            let config = getConfig(gameName);
            let url = utils_1.kdutils.format("https://api.weixin.qq.com/cgi-bin/token?grant_type=client_credential&appid={0}&secret={1}", config.appId, config.appSecret);
            let t = yield reqGet(url);
            info.waiting = false;
            if (t == null) {
                return null;
            }
            log_1.Log.oth.info("...", t);
            info.ak = t.access_token;
            info.expire = (t.expires_in || 7200 - 60) * 1000;
            if (info.expire < 0) {
                info.expire = 10000;
            }
            info.startTime = utils_1.kdutils.getMillionSecond();
            log_1.Log.oth.info("[wechat] get ak success gameName = " + gameName + " expire = " + info.expire + " ak = " + info.ak);
            return info.ak;
        });
    }
    WechatService.getAK = getAK;
    function msgSecCheck(gameName, content) {
        return __awaiter(this, void 0, void 0, function* () {
            let ak = yield getAK(gameName);
            if (ak == null) {
                return true;
            }
            //content = "特3456书yuuo莞6543李zxcz蒜7782法fgnv级"
            let url = "https://api.weixin.qq.com/wxa/msg_sec_check?access_token=" + ak;
            let ret = yield req_1.kdreq.postJson(url, {
                content: content
            });
            //Log.oth.info("body = ", ret.body)
            if (ret.error) {
                log_1.Log.oth.error("[wechat] msg sec post failed error = ", ret.error);
                return true;
            }
            if (ret.body == null) {
                log_1.Log.oth.error("[wechat] msg sec post failed body is null ret = ", ret);
                return true;
            }
            if (ret.body.errcode == 87014) {
                return false;
            }
            if (ret.body.errcode != 0) {
                log_1.Log.oth.error("[wechat] msg sec failed body = ", ret.body);
                return true;
            }
            return true;
        });
    }
    WechatService.msgSecCheck = msgSecCheck;
    function msgSecCheckNew(gameName, content, openid) {
        return __awaiter(this, void 0, void 0, function* () {
            let ak = yield getAK(gameName);
            if (ak == null) {
                return true;
            }
            //content = "特3456书yuuo莞6543李zxcz蒜7782法fgnv级"
            let url = "https://api.weixin.qq.com/wxa/msg_sec_check?access_token=" + ak;
            let ret = yield req_1.kdreq.postJson(url, {
                content: content,
                openid: openid,
            });
            //Log.oth.info("body = ", ret.body)
            if (ret.errcode != 0) {
                log_1.Log.oth.error("[wechat] msg sec post failed error = ", ret.errmsg);
                return true;
            }
            if (!ret.result) {
                log_1.Log.oth.error("[wechat] msg sec post failed result is null ret = ", ret);
                return true;
            }
            if (ret.result.label != 100) {
                log_1.Log.oth.error("[wechat] msg sec failed label = ", ret.result.label);
                return false;
            }
            return true;
        });
    }
    WechatService.msgSecCheckNew = msgSecCheckNew;
    function mediaCheck(gameName, content, openid, media_type, userid) {
        return __awaiter(this, void 0, void 0, function* () {
            let ak = yield getAK(gameName);
            if (ak == null) {
                return true;
            }
            //content = "特3456书yuuo莞6543李zxcz蒜7782法fgnv级"
            let url = "https://api.weixin.qq.com/wxa/media_check_async?access_token=" + ak;
            let ret = yield req_1.kdreq.postJson(url, {
                media_url: content,
                openid: openid,
                media_type: media_type
            });
            log_1.Log.oth.info("media_check_async callback =>", JSON.stringify(ret));
            if (ret.errcode != 0) {
                log_1.Log.oth.error("[wechat] msg sec post failed error = ", ret.errmsg);
                return ret;
            }
            traceList.push({ trace_id: ret.trace_id, userid: userid, iconUrl: content });
            return ret;
        });
    }
    WechatService.mediaCheck = mediaCheck;
    function imageCheck(event) {
        return __awaiter(this, void 0, void 0, function* () {
            console.log('检测返回数据包 =>', JSON.stringify(event));
            log_1.Log.oth.info('检测返回数据包 =>', JSON.stringify(event));
        });
    }
    WechatService.imageCheck = imageCheck;
    let activityTableName = "t_wechat_activity_id";
    function getActivityKeyName(gameName, id) {
        return gameName + " | " + id;
    }
    WechatService.getActivityKeyName = getActivityKeyName;
    function createActivityId(gameName) {
        return __awaiter(this, void 0, void 0, function* () {
            let ak = yield getAK(gameName);
            if (ak == null) {
                return null;
            }
            let url = "https://api.weixin.qq.com/cgi-bin/message/wxopen/activityid/create?access_token=" + ak;
            let ret = yield reqGet(url);
            if (ret == null) {
                return null;
            }
            let info = {
                id: ret.activity_id,
                expire: ret.expiration_time,
                startTime: utils_1.kdutils.getMillionSecond(),
            };
            yield rpc_1.Rpc.center.callException("kds.dbp.kv.sett", activityTableName, getActivityKeyName(gameName, info.id), JSON.stringify(info));
            return info.id;
        });
    }
    WechatService.createActivityId = createActivityId;
    function getActivityInfo(gameName, id) {
        return __awaiter(this, void 0, void 0, function* () {
            let str = yield rpc_1.Rpc.center.callException("kds.dbp.kv.gett", activityTableName, getActivityKeyName(gameName, id));
            if (str == null) {
                return null;
            }
            let t = JSON.parse(str);
            return t;
        });
    }
    WechatService.getActivityInfo = getActivityInfo;
    function removeActivityInfo(gameName, id) {
        return __awaiter(this, void 0, void 0, function* () {
            yield rpc_1.Rpc.center.callException("kds.dbp.kv.delt", activityTableName, getActivityKeyName(gameName, id));
            return true;
        });
    }
    WechatService.removeActivityInfo = removeActivityInfo;
    function updateActivityInfo(gameName, id, params) {
        return __awaiter(this, void 0, void 0, function* () {
            let ak = yield getAK(gameName);
            if (ak == null) {
                return false;
            }
            let info = yield getActivityInfo(gameName, id);
            if (info == null) {
                return false;
            }
            if (utils_1.kdutils.getMillionSecond() - info.startTime >= info.expire) {
                log_1.Log.oth.info("[wechat] activity id expired info = ", info);
                rpc_1.Rpc.center.callException("kds.dbp.kv.delt", activityTableName, getActivityKeyName(gameName, id));
                return false;
            }
            let url = "https://api.weixin.qq.com/cgi-bin/message/wxopen/updatablemsg/send?access_token=" + ak;
            let paramList = [
                { name: "member_count", value: params.member_count },
                { name: "room_limit", value: params.room_limit },
            ];
            if (params.state == 1) {
                paramList.push({
                    name: "path", value: "?cmd=enter&room_id=" + params.roomID
                });
                paramList.push({
                    name: "version_type", value: config_1.Config.localConfig.wechatVersionType || "release",
                });
            }
            log_1.Log.oth.info("[wechat] update active id = " + info.id + " | url = " + url, params);
            let postRet = yield req_1.kdreq.postJson(url, {
                access_token: ak,
                activity_id: info.id,
                target_state: params.state,
                template_info: {
                    parameter_list: paramList
                }
            });
            if (postRet.error || postRet.body == null) {
                log_1.Log.oth.error("[wechat] update activity failed ", postRet);
                return false;
            }
            if (postRet.body.errcode != 0) {
                log_1.Log.oth.error("[wechat] update activity error ", postRet.body);
                return false;
            }
            return true;
        });
    }
    WechatService.updateActivityInfo = updateActivityInfo;
    function setUserSessionKey(openID, key) {
        return __awaiter(this, void 0, void 0, function* () {
            yield rpc_1.Rpc.center.callException("kds.dbp.kv.sett", KVDefine_1.KVDefine.WXOpenID2SSKeyTablename, KVDefine_1.KVDefine.WXOpenID2SSKeyPrefix + openID, key);
        });
    }
    WechatService.setUserSessionKey = setUserSessionKey;
    function getUserSessionKey(openID) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield rpc_1.Rpc.center.callException("kds.dbp.kv.gett", KVDefine_1.KVDefine.WXOpenID2SSKeyTablename, KVDefine_1.KVDefine.WXOpenID2SSKeyPrefix + openID);
        });
    }
    WechatService.getUserSessionKey = getUserSessionKey;
    function getUserOpenID(userID) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield rpc_1.Rpc.center.callException("kds.dbp.kv.gett", KVDefine_1.KVDefine.WXUserID2OpenIDTablename, KVDefine_1.KVDefine.WXUserID2OpenIDPrefix + userID);
        });
    }
    WechatService.getUserOpenID = getUserOpenID;
    function getUserSessionKeyByUserID(userID) {
        return __awaiter(this, void 0, void 0, function* () {
            let openID = yield getUserOpenID(userID);
            if (openID) {
                return yield getUserSessionKey(openID);
            }
            return null;
        });
    }
    WechatService.getUserSessionKeyByUserID = getUserSessionKeyByUserID;
    let urls;
    (function (urls) {
        urls.code2Session = "https://api.weixin.qq.com/sns/jscode2session?appid={0}&secret={1}&js_code={2}&grant_type=authorization_code";
    })(urls || (urls = {}));
    function loginWithCode(gameName, code) {
        return __awaiter(this, void 0, void 0, function* () {
            return new Promise(function (resolve, reject) {
                let config = config_1.Config.localConfig.games.find(v => v.gameName == gameName);
                if (config == null) {
                    log_1.Log.oth.info("login: cannot find game config name = " + gameName);
                    return null;
                }
                let url = utils_1.kdutils.format(urls.code2Session, config.appId, config.appSecret, code);
                log_1.Log.oth.info("[WechatLoginService] req url = " + url);
                _request(url, function (error, response, body) {
                    if (!error && response.statusCode == 200) {
                        log_1.Log.oth.info("[WechatLoginService] req body = " + body + " | type = " + typeof (body));
                        let t = null;
                        try {
                            t = JSON.parse(body);
                        }
                        catch (error) {
                            log_1.Log.oth.error("[WechatLoginService] parse json failed ", error);
                            resolve(null);
                            return;
                        }
                        if (t.errcode != null && t.errcode != 0) {
                            log_1.Log.oth.error("[WechatLoginService] req url failed with code url = " + url + " | code = " + t.errcode + " | msg = " + t.errmsg);
                            resolve(null);
                            return;
                        }
                        log_1.Log.oth.info("[WechatLoginService] return msg: ", t);
                        let ret = {
                            openId: t.openid ? t.openid : t.unionid,
                            session_key: t.session_key
                        };
                        setUserSessionKey(ret.openId, ret.session_key)
                            .then(function () {
                            resolve(ret);
                        });
                    }
                    else {
                        log_1.Log.oth.error("[WechatLoginService] req failed ", error);
                        resolve(null);
                    }
                });
            });
        });
    }
    WechatService.loginWithCode = loginWithCode;
})(WechatService = exports.WechatService || (exports.WechatService = {}));
